#!/usr/bin/env python3
"""
CodeLink Installer - Double-click this file.
"""

import os
import subprocess
import sys
from pathlib import Path


def safe_print(text):
    try:
        print(text)
    except UnicodeEncodeError:
        print(text.encode("ascii", errors="replace").decode("ascii"))


def main():
    safe_print("")
    safe_print("  ==================================")
    safe_print("       CodeLink - Installer")
    safe_print("  ==================================")
    safe_print("")

    if sys.version_info < (3, 9):
        safe_print("  [Error] Python 3.9+ required (you have %s)" % sys.version)
        safe_print("  Download: https://python.org/downloads")
        input("\n  Press Enter to exit...")
        sys.exit(1)

    safe_print("  [OK] Python " + sys.version.split()[0])
    safe_print("")
    safe_print("  Installing dependencies...")

    req = Path(__file__).parent / "requirements.txt"
    r = subprocess.run(
        [sys.executable, "-m", "pip", "install", "-q", "-r", str(req)],
        capture_output=True, text=True,
    )
    if r.returncode != 0:
        r = subprocess.run(
            [sys.executable, "-m", "pip", "install", "-q", "--user", "-r", str(req)],
            capture_output=True, text=True,
        )

    if r.returncode == 0:
        safe_print("  [OK] Dependencies installed")
    else:
        safe_print("  [Error] Install failed:")
        safe_print(r.stderr)
        input("\n  Press Enter to exit...")
        sys.exit(1)

    # Run setup + start
    safe_print("")
    app = Path(__file__).parent / "codelink.py"
    subprocess.run([sys.executable, str(app)])


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        safe_print("\n  Cancelled.")
    except Exception as e:
        safe_print("\n  [Error] " + str(e))
        input("\n  Press Enter to exit...")
