#!/usr/bin/env python3
"""
CodeLink - Claude Code on Telegram.
Your laptop. Your AI. Your phone.

No user IDs. No tokens. Just a code.
"""

import json
import logging
import os
import platform
import shutil
import subprocess
import sys
import time
import uuid
import webbrowser
from pathlib import Path

import requests

logging.basicConfig(
    format="%(asctime)s [CodeLink] %(message)s",
    level=logging.INFO,
)
logger = logging.getLogger("codelink")


def safe_print(text):
    try:
        print(text)
    except UnicodeEncodeError:
        print(text.encode("ascii", errors="replace").decode("ascii"))


# -- config ----------------------------------------------------------------
CONFIG_DIR = Path.home() / ".codelink"
CONFIG_FILE = CONFIG_DIR / "config.json"

SERVER_URL = "http://167.71.111.85:8642"


def load_config():
    if CONFIG_FILE.exists():
        return json.loads(CONFIG_FILE.read_text())
    return {}


def save_config(cfg):
    CONFIG_DIR.mkdir(parents=True, exist_ok=True)
    CONFIG_FILE.write_text(json.dumps(cfg, indent=2))


# -- claude code -----------------------------------------------------------

def find_claude():
    which = shutil.which("claude")
    if which:
        return which
    candidates = [
        os.path.expanduser("~/.claude/local/claude"),
        os.path.expanduser("~/.local/bin/claude"),
        "/usr/local/bin/claude",
    ]
    if platform.system() == "Windows":
        appdata = os.environ.get("APPDATA", "")
        localappdata = os.environ.get("LOCALAPPDATA", "")
        candidates += [
            os.path.join(appdata, "Claude", "claude.exe"),
            os.path.join(localappdata, "Programs", "claude", "claude.exe"),
        ]
    for p in candidates:
        if os.path.isfile(p):
            return p
    return "claude"


CLAUDE = find_claude()


def run_claude(prompt, workdir, timeout=300):
    cmd = [CLAUDE, "--print", "--no-input", prompt]
    try:
        result = subprocess.run(
            cmd, capture_output=True, text=True, cwd=workdir, timeout=timeout,
        )
        out = result.stdout.strip()
        if result.returncode != 0 and not out:
            return "[Error] Something went wrong. Try again."
        return out or "[Done]"
    except subprocess.TimeoutExpired:
        return "[Timeout] That took too long. Try a shorter request."
    except FileNotFoundError:
        return (
            "[Error] Claude Code not found on this machine.\n\n"
            "Install it: https://docs.anthropic.com/en/docs/claude-code\n"
            "Then run CodeLink again."
        )
    except Exception as e:
        logger.error("Error: %s", e)
        return "[Error] Something went wrong. Try again."


# -- client ----------------------------------------------------------------

class CodeLinkClient:
    def __init__(self, config):
        self.chat_id = config.get("chat_id")
        self.device_id = config.get("device_id", str(uuid.uuid4()))
        self.api_key = config.get("api_key")
        self.workdir = config.get("workdir", os.getcwd())
        self.timeout = config.get("timeout", 300)
        self.running = True

    def headers(self):
        return {"Authorization": "Bearer " + (self.api_key or "")}

    def poll(self):
        try:
            r = requests.get(
                SERVER_URL + "/poll/" + str(self.chat_id),
                headers=self.headers(), timeout=10,
            )
            if r.status_code == 200:
                return r.json().get("messages", [])
            return []
        except:
            return []

    def respond(self, msg_id, response):
        try:
            requests.post(
                SERVER_URL + "/respond/" + str(self.chat_id),
                headers=self.headers(),
                json={"msg_id": msg_id, "response": response},
                timeout=10,
            )
        except Exception as e:
            logger.error("Respond error: %s", e)

    def heartbeat(self):
        try:
            requests.post(
                SERVER_URL + "/heartbeat/" + str(self.chat_id),
                headers=self.headers(), timeout=5,
            )
        except:
            pass

    def run(self):
        safe_print("")
        safe_print("  ==================================")
        safe_print("       CodeLink - Running")
        safe_print("  ==================================")
        safe_print("")
        safe_print("  Claude:  " + CLAUDE)
        safe_print("  Folder:  " + self.workdir)
        safe_print("")
        safe_print("  [OK] Connected to server!")
        safe_print("  Open Telegram and message @CodeLinkBot.")
        safe_print("  Press Ctrl+C to stop.")
        safe_print("")

        last_heartbeat = 0

        while self.running:
            try:
                if time.time() - last_heartbeat > 10:
                    self.heartbeat()
                    last_heartbeat = time.time()

                messages = self.poll()

                for msg in messages:
                    msg_id = msg["id"]
                    text = msg["text"]

                    safe_print("  >> " + text[:80])

                    response = run_claude(text, self.workdir, self.timeout)

                    safe_print("  << " + response[:80])

                    self.respond(msg_id, response)

                time.sleep(1)

            except KeyboardInterrupt:
                self.running = False
            except Exception as e:
                logger.error("Error: %s", e)
                time.sleep(5)

        safe_print("\n  CodeLink stopped.\n")


# -- linking ---------------------------------------------------------------

def link_device(device_id):
    """Get a code from server, show it to user, wait for them to enter it in Telegram."""

    safe_print("  Connecting to server...")

    try:
        r = requests.post(
            SERVER_URL + "/request-code",
            json={"device_id": device_id},
            timeout=10,
        )
    except requests.ConnectionError:
        safe_print("  [Error] Can't reach server. Check your internet.")
        return None
    except Exception as e:
        safe_print("  [Error] " + str(e))
        return None

    if r.status_code != 200:
        safe_print("  [Error] Server error: " + r.text)
        return None

    data = r.json()
    code = data["code"]
    api_key = data["api_key"]

    safe_print("")
    safe_print("  ==========================================")
    safe_print("  Your link code:  " + code)
    safe_print("  ==========================================")
    safe_print("")
    safe_print("  Now open Telegram and send this code to @CodeLinkBot")
    safe_print("")

    # Try to open Telegram bot link in browser
    try:
        webbrowser.open("https://t.me/CodeLinkBot")
    except:
        pass

    safe_print("  Waiting for you to enter the code in Telegram...")
    safe_print("")

    # Poll until linked or timeout (5 minutes)
    start = time.time()
    while time.time() - start < 300:
        try:
            r = requests.get(SERVER_URL + "/check-code/" + code, timeout=10)
            if r.status_code == 200:
                result = r.json()
                if result.get("linked"):
                    return {
                        "chat_id": result["chat_id"],
                        "api_key": api_key,
                    }
                if result.get("expired"):
                    safe_print("  [Error] Code expired. Run again.")
                    return None
        except:
            pass
        time.sleep(2)

    safe_print("  [Error] Timed out waiting for code. Run again.")
    return None


# -- setup -----------------------------------------------------------------

def setup():
    safe_print("")
    safe_print("  ==================================")
    safe_print("       CodeLink - Setup")
    safe_print("  ==================================")
    safe_print("")

    claude_ok = shutil.which("claude") is not None
    if claude_ok:
        safe_print("  [OK] Claude Code found")
    else:
        safe_print("  [!] Claude Code not detected")
        safe_print("      Install: https://docs.anthropic.com/en/docs/claude-code")
        safe_print("      (you can install it later)")

    safe_print("")
    default_wd = os.getcwd()
    wd = input("  Working folder [" + default_wd + "]: ").strip()
    if not wd:
        wd = default_wd
    wd = os.path.abspath(os.path.expanduser(wd))

    if not os.path.isdir(wd):
        os.makedirs(wd, exist_ok=True)

    device_id = str(uuid.uuid4())

    # Link with Telegram
    safe_print("")
    link_result = link_device(device_id)

    if not link_result:
        input("\n  Press Enter to exit...")
        sys.exit(1)

    safe_print("  [OK] Linked successfully!")
    safe_print("")

    cfg = {
        "chat_id": link_result["chat_id"],
        "api_key": link_result["api_key"],
        "device_id": device_id,
        "workdir": wd,
        "timeout": 300,
        "setup_done": True,
    }
    save_config(cfg)

    return cfg


def main():
    cfg = load_config()

    if "--setup" in sys.argv or not cfg.get("setup_done"):
        cfg = setup()

    client = CodeLinkClient(cfg)
    client.run()


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        safe_print("\n\n  Stopped.\n")
    except Exception as e:
        safe_print("\n  [Error] " + str(e))
        input("\n  Press Enter to exit...")
